﻿using System.Collections;
using UnityEngine;

using UnityEngine.SceneManagement;  // Used for switching scenes

//////////////////////////////////////////////////////////////////
// [Class for enemy auto-generation, Game Over / Game Clear, and restart] 
// Handles random enemy auto-generation based on Fighter level,
// Restart processing, and Game Over / Game Clear processing
//////////////////////////////////////////////////////////////////
public class GameController : MonoBehaviour
{
    /////////////////////////////////
    // [Variable settings]
    /////////////////////////////////

    // [Constant settings]
    const float TIME_ENEMY = 2f;  // Base interval for enemy appearance
    const float BOSS_TIME = 90;  // Time until the boss appears

    // [Public variables accessed from other classes]
    public bool isUpperL3 = false;  // Flag indicating Fighter is level 3 or higher
    public bool isFinish = false;  // Game finished flag

    // [Variables set from the Unity Editor]
    [SerializeField] GameObject pre_EGreen1;  // Prefab for Green1 enemy
    [SerializeField] GameObject pre_EGreen2;  // Prefab for Green2 enemy
    [SerializeField] GameObject pre_EYellow1;  // Prefab for Yellow group 1
    [SerializeField] GameObject pre_EYellow2;  // Prefab for Yellow group 2
    [SerializeField] GameObject pre_ERed1;  // Prefab for Red1 enemy
    [SerializeField] GameObject pre_ERed2;  // Prefab for Red2 enemy

    [SerializeField] GameObject go_Boss;  // Boss game object
    [SerializeField] GameObject go_GameClear;  // Game Clear UI
    [SerializeField] GameObject go_GameOver;  // Game Over UI

    // [Private variables used in this program]
    bool isBossMode = false;  // Boss-appearance mode (initial value is false)
    float time = 0;  // Elapsed game time


    /////////////////////////////////
    // [Function settings]
    /////////////////////////////////

    // [Function that runs only once at game start or when the game object is generated]
    // (Unity standard function)
    void Start()
    {
        // Call coroutine function Generate_Enemy
        StartCoroutine(Generate_Enemy());
    }


    // [Function that runs once per frame]
    // (Unity standard function)
    void Update()
    {
        // Calculate elapsed game time by adding the time per frame
        time += Time.deltaTime;

        // If not in Boss mode and elapsed time exceeds boss appearance time
        if (!isBossMode && time > BOSS_TIME)
        {
            // Set boss-appearance mode to true
            isBossMode = true;
            // Activate the boss game object
            go_Boss.SetActive(true);
        }
    }


    // [Coroutine to generate enemies]
    IEnumerator Generate_Enemy()
    {
        // Get random time until the next enemy appears
        float time_Wait = Random.Range(TIME_ENEMY - 0.7f, TIME_ENEMY);

        // Random X-coordinate where the enemy will be generated
        float pos_Spoon_X = Random.Range(-2.4f, 2.4f);

        GameObject pre_Enemy;  // Enemy game object to be generated
        // Get a random number from 1 to 6
        switch (Random.Range(1, 7))
        {
            // With a 3/6 chance
            case 1:
            case 2:
            case 3:
                // If the Fighter level-3 flag is false, generate pre_EGreen1
                if (!isUpperL3) pre_Enemy = pre_EGreen1;
                // If the Fighter level-3 flag is true, generate pre_EGreen2
                else pre_Enemy = pre_EGreen2;
                break;

            // With a 2/6 chance
            case 4:
            case 5:
                // If the Fighter level-3 flag is false, generate pre_EYellow1
                if (!isUpperL3) pre_Enemy = pre_EYellow1;
                // If the Fighter level-3 flag is true, generate pre_EYellow2
                else pre_Enemy = pre_EYellow2;
                break;

            // With a 1/6 chance
            default:
                // If the Fighter level-3 flag is false
                if (!isUpperL3)
                {
                    // If 15 seconds have elapsed, generate pre_ERed1
                    if (time > 15) pre_Enemy = pre_ERed1;
                    // If less than 15 seconds have elapsed, generate pre_EGreen1
                    else pre_Enemy = pre_EGreen1;
                }
                // If the Fighter level-3 flag is true, generate pre_ERed2
                else pre_Enemy = pre_ERed2;
                break;
        }

        // Generate the chosen enemy at a random position at the top of the screen
        Instantiate(pre_Enemy,
            new Vector3(pos_Spoon_X, 5.5f, 0), Quaternion.identity);

        // Wait for the specified number of seconds
        yield return new WaitForSeconds(time_Wait);

        // If not in Boss mode and the game is not finished, call Generate_Enemy again
        if (!isBossMode && !isFinish) StartCoroutine(Generate_Enemy());
    }


    // [Game Clear processing function]
    public void OnGameClear()
    {
        // Set the game finished flag to true
        isFinish = true;
        // Enable the Game Clear Text UI game object
        go_GameClear.SetActive(true);

        // Call coroutine function ReStart
        StartCoroutine(ReStart());
    }


    // [Game Over processing function]
    public void OnGameOver()
    {
        // Set the game finished flag to true
        isFinish = true;
        // Enable the Game Over Text UI game object
        go_GameOver.SetActive(true);

        // Call coroutine function ReStart
        StartCoroutine(ReStart());
    }


    // [Coroutine to Restart processing]
    IEnumerator ReStart()
    {
        // Wait for the specified number of seconds
        yield return new WaitForSeconds(10);

        // Load the "GameMain" scene
        SceneManager.LoadScene("GameMain");
    }
}
