﻿using System.Collections;
using UnityEngine;

using UnityEngine.UI;  // Used for UI control processing

//////////////////////////////////////////////////////////////////
// [Class for Fighter movement, collision detection, and beam firing] 
// Handles mouse-based movement, beam firing based on beam type and Fighter level, 
// Collision detection, HP/score/level control, and UI control
//////////////////////////////////////////////////////////////////
public class FighterController : MonoBehaviour
{
    /////////////////////////////////
    // [Variable settings]
    /////////////////////////////////

    // [Constant settings]
    const int MAX_HP = 20;  // Initial Fighter HP / Max HP
    const int LEVELUP_SCORE = 20;  // Score needed for Fighter level-up

    // [Public variables accessed from other classes]
    public int score = 0;  // Score (added by defeated enemies)

    // [Variables set from the Unity Editor]
    [SerializeField] GameObject pre_Beam1;  // Prefab for Fighter Beam1
    [SerializeField] GameObject pre_Beam2;  // Prefab for Fighter Beam2
    [SerializeField] GameObject beam2_L3;  // Parent object for Beam2 at level 3
    [SerializeField] GameObject beam2_L4;  // Parent object for Beam2 at level 4

    [SerializeField] Text textScore;  // On-screen Text UI for score
    [SerializeField] Text textLevel;  // On-screen Text UI for level
    [SerializeField] Text textHP;  // On-screen Text UI for HP

    [SerializeField] Sprite[] sprite;  // Fighter sprites for each level
    [SerializeField] SpriteRenderer sRenderer;  // SpriteRenderer
    [SerializeField] GameController gameController;  // GameController

    // [Private variables used in this program]
    int fighter_HP = MAX_HP;  // Fighter HP
    int fighter_LV = 1;  // Fighter level

    Vector3 startPos;  // Initial mouse position on-screen per frame for movement


    /////////////////////////////////
    // [Function settings]
    /////////////////////////////////

    // [Function that runs only once at game start or when the game object is generated]
    // (Unity standard function)
    void Start()
    {
        // Call coroutine function Create_Beam1
        StartCoroutine(Create_Beam1());

        // Display score in Text UI
        textScore.text = score.ToString();
        // Display level in Text UI
        textLevel.text = fighter_LV.ToString();
        // Display HP in Text UI
        textHP.text = fighter_HP.ToString();
    }


    // [Function that runs once every frame]
    // (Unity standard function)
    void Update()
    {
        // When the game has finished
        if (gameController.isFinish)
        {
            // Delete the parent object for Beam2 at level 3
            Destroy(beam2_L3);
            // Delete the parent object for Beam2 at level 4
            Destroy(beam2_L4);
        }

        // When the score is greater than or equal to the level-up score
        if (score >= LEVELUP_SCORE)
        {
            // Reset score to 0
            score = 0;
            // Reset HP to max HP
            fighter_HP = MAX_HP;

            // Display HP in Text UI
            textHP.text = fighter_HP.ToString();

            // When the level is 3 or less
            if (fighter_LV <= 3)
            {
                // Increase the level
                fighter_LV++;
                // Display level in Text UI
                textLevel.text = fighter_LV.ToString();

                // Change Fighter sprite to the sprite for the increased level
                sRenderer.sprite = sprite[fighter_LV - 1];

                // When the level reaches 3
                if (fighter_LV == 3)
                {
                    // Execute the level-3 Beam2 firing function
                    CreateL3_Beam2();
                    // Set GameController.isUpperL3 to true
                    gameController.isUpperL3 = true;
                }

                // When the level reaches 4, execute the level-4 Beam2 firing function
                if (fighter_LV == 4) CreateL4_Beam2();
            }
        }

        // Update Text UI Score (score is updated by enemies, so refresh every frame)
        textScore.text = score.ToString();

        // When the left mouse button is pressed
        if (Input.GetMouseButtonDown(0))
        {
            // Get the current mouse cursor position on-screen
            startPos = Camera.main.ScreenToWorldPoint(Input.mousePosition);
        }

        // When the left mouse button is held down
        if (Input.GetMouseButton(0))
        {
            // Get the current mouse position on-screen (end position for this frame)
            Vector3 endPos = Camera.main.ScreenToWorldPoint(Input.mousePosition);

            // Mouse movement vector for this frame (mouse end minus start position)
            Vector3 difPos = endPos - startPos;

            // If Fighter position plus the mouse movement remains within the screen
            if (2.4f > transform.position.x + difPos.x &&
                    transform.position.x + difPos.x > -2.4f &&
                1 > transform.position.y + difPos.y &&
                    transform.position.y + difPos.y > -4.5f)
            {
                // Move the Fighter by the mouse movement vector for this frame
                transform.Translate(difPos);
            }

            // To calculate the mouse movement of next frame, get the mouse position
            startPos = Camera.main.ScreenToWorldPoint(Input.mousePosition);
        }
    }


    // [Function called when the Fighter’s collider hits another collider]
    // (Unity standard function)
    private void OnTriggerEnter2D(Collider2D collision)
    {
        // When the tag of the object hit is "Beam1_E" or "Beam2_E"
        if (collision.gameObject.tag == "Beam1_E"
            || collision.gameObject.tag == "Beam2_E")
        {
            // Decrease the Fighter’s HP by 1
            fighter_HP--;
            // Display HP in Text UI
            textHP.text = fighter_HP.ToString();

            // When the tag of the object hit is "Beam1_E"
            if (collision.gameObject.tag == "Beam1_E")
                // Delete the enemy Beam1 that hit
                Destroy(collision.gameObject);
        }

        // When the tag of the object hit is "Beam3_E"
        if (collision.gameObject.tag == "Beam3_E")
        {
            // Decrease the Fighter’s HP by 10
            fighter_HP -= 10;
            // Display HP in Text UI
            textHP.text = fighter_HP.ToString();

            // Delete the beam that hit
            Destroy(collision.gameObject);
        }

        // After HP is reduced, when HP is 0 or less
        if (fighter_HP <= 0)
        {
            // When the level is greater than 1
            if (fighter_LV > 1)
            {
                // Reset score to 0
                score = 0;
                // Reset HP to max HP
                fighter_HP = MAX_HP;
                // Decrease the level by 1
                fighter_LV--;

                // Display HP in Text UI
                textHP.text = fighter_HP.ToString();
                // Display level in Text UI
                textLevel.text = fighter_LV.ToString();
                // Change Fighter sprite to the sprite for the decreased level
                sRenderer.sprite = sprite[fighter_LV - 1];

                // When the decreased level is 2
                if (fighter_LV == 2)
                {
                    // Execute the level-3 Beam2 deletion function
                    DeleteL3_Beam2();
                    // Set GameController.isUpperL3 to false
                    gameController.isUpperL3 = false;
                }

                // When the decreased level is 3, call level-4 Beam2 deletion function
                if (fighter_LV == 3) DeleteL4_Beam2();
            }

            // When the level is 1
            else
            {
                // Call the Game Over function in GameController
                gameController.OnGameOver();
                // Delete the Fighter
                Destroy(this.gameObject);
            }
        }
    }


    // [Coroutine to generate Beam1]
    IEnumerator Create_Beam1()
    {
        Vector3 pos = transform.position;  // Fighter position at the firing timing

        // Generate Beam1 (pre_Beam1) in the scene (upper-right of Fighter, upward)
        Instantiate(pre_Beam1,
            new Vector3(pos.x + 0.4f, pos.y + 0.4f, pos.z), Quaternion.identity);
        // Generate Beam1 (pre_Beam1) in the scene (upper-left of Fighter, upward)
        Instantiate(pre_Beam1,
            new Vector3(pos.x - 0.4f, pos.y + 0.4f, pos.z), Quaternion.identity);

        // When level is 2, 3, or 4
        if (fighter_LV == 2 || fighter_LV == 3 || fighter_LV == 4)
        {
            // Generate Beam1 (pre_Beam1) (upper-left of Fighter, diagonally left)
            Instantiate(pre_Beam1, new Vector3(pos.x - 0.3f, pos.y + 0.3f, pos.z),
                Quaternion.Euler(0, 0, 30));
            // Generate Beam1 (pre_Beam1) (upper-right of Fighter, diagonally right)
            Instantiate(pre_Beam1, new Vector3(pos.x + 0.3f, pos.y + 0.3f, pos.z),
                Quaternion.Euler(0, 0, -30));

            // Generate Beam1 (pre_Beam1) in the scene (above Fighter, upward)
            Instantiate(pre_Beam1, new Vector3(pos.x, pos.y + 0.4f, pos.z),
                Quaternion.identity);
        }

        // When level is 3 or 4
        if (fighter_LV == 3 || fighter_LV == 4)
        {
            // Generate Beam1 (pre_Beam1) (lower-left of Fighter, diagonally left)
            Instantiate(pre_Beam1, new Vector3(pos.x - 0.3f, pos.y - 0.3f, pos.z),
                Quaternion.Euler(0, 0, 150));
            // Generate Beam1 (pre_Beam1) (lower-right of Fighter, diagonally right)
            Instantiate(pre_Beam1, new Vector3(pos.x + 0.3f, pos.y - 0.3f, pos.z),
                Quaternion.Euler(0, 0, -150));
            // Generate Beam1 (pre_Beam1) in the scene (below Fighter, downward)
            Instantiate(pre_Beam1, new Vector3(pos.x, pos.y - 0.4f, pos.z),
                Quaternion.Euler(0, 0, 180));
        }

        // Wait for the specified number of seconds
        yield return new WaitForSeconds(0.2f);

        // If the game is not finished, call coroutine function Create_Beam1 again
        if (!gameController.isFinish) StartCoroutine(Create_Beam1());
    }


    // [Beam2 firing function when level is 3]
    void CreateL3_Beam2()
    {
        Vector3 pos = transform.position; // Fighter position at the moment of firing
        // Start the function that generates 4 Beam2 instances
        Create_Beam2(pos, beam2_L3);
    }


    // [Beam2 firing function when level is 4]
    void CreateL4_Beam2()
    {
        Vector3 pos = transform.position;  // Fighter position at the firing timing

        // Get position at upper-right of Fighter
        Vector3 pos1 = new Vector3(pos.x + 1.5f, pos.y + 1.5f, pos.z);
        // Start function that generates 4 Beam2 instances centered on this position
        Create_Beam2(pos1, beam2_L4);
        // Get position at upper-left of Fighter
        Vector3 pos2 = new Vector3(pos.x - 1.5f, pos.y + 1.5f, pos.z);
        // Start function that generates 4 Beam2 instances centered on this position
        Create_Beam2(pos2, beam2_L4);
        // Get position at lower-right of Fighter
        Vector3 pos3 = new Vector3(pos.x + 1.5f, pos.y - 1.5f, pos.z);
        // Start function that generates 4 Beam2 instances centered on this position
        Create_Beam2(pos3, beam2_L4);
        // Get position at lower-left of Fighter
        Vector3 pos4 = new Vector3(pos.x - 1.5f, pos.y - 1.5f, pos.z);
        // Start function that generates 4 Beam2 instances centered on this position
        Create_Beam2(pos4, beam2_L4);
    }


    // [Function that generates 4 Beam2 instances centered on a specified position]
    void Create_Beam2(Vector3 pos, GameObject parent_GO)
    {
        // Generate Beam2 (pre_Beam2) in the scene (above the rotation center)
        GameObject beamA = Instantiate(pre_Beam2,
            new Vector3(pos.x, pos.y + 1, pos.z), Quaternion.Euler(0, 0, 90));
        // Set the rotation angle that changes the beam’s direction of travel
        beamA.GetComponent<BeamController>().beam_Angle = 200;
        // Set the generated beam’s game object under the specified parent object
        beamA.transform.parent = parent_GO.transform;

        // Generate Beam2 (pre_Beam2) in the scene (left of the rotation center)
        GameObject beamB = Instantiate(pre_Beam2,
            new Vector3(pos.x - 1, pos.y, pos.z), Quaternion.Euler(0, 0, 180));
        // Set the rotation angle that changes the beam’s direction of travel
        beamB.GetComponent<BeamController>().beam_Angle = 200;
        // Set the generated beam’s game object under the specified parent object
        beamB.transform.parent = parent_GO.transform;

        // Generate Beam2 (pre_Beam2) in the scene (below the rotation center)
        GameObject beamC = Instantiate(pre_Beam2,
            new Vector3(pos.x, pos.y - 1, pos.z), Quaternion.Euler(0, 0, -90));
        // Set the rotation angle that changes the beam’s direction of travel
        beamC.GetComponent<BeamController>().beam_Angle = 200;
        // Set the generated beam’s game object under the specified parent object
        beamC.transform.parent = parent_GO.transform;

        // Generate Beam2 (pre_Beam2) in the scene (right of the rotation center)
        GameObject beamD = Instantiate(pre_Beam2,
            new Vector3(pos.x + 1, pos.y, pos.z), Quaternion.Euler(0, 0, 0));
        // Set the rotation angle that changes the beam’s direction of travel
        beamD.GetComponent<BeamController>().beam_Angle = 200;
        // Set the generated beam’s game object under the specified parent object
        beamD.transform.parent = parent_GO.transform;
    }


    // [Beam2 deletion function when level 3 is lost]
    void DeleteL3_Beam2()
    {
        // Delete all Beam2 instances generated under Beam2_L3
        foreach (Transform tf in beam2_L3.transform)
        {
            Destroy(tf.gameObject);
        }
    }


    // [Beam2 deletion function when level 4 is lost]
    void DeleteL4_Beam2()
    {
        // Delete all Beam2 instances generated under Beam2_L4
        foreach (Transform tf in beam2_L4.transform)
        {
            Destroy(tf.gameObject);
        }
    }
}
