﻿using System.Collections;
using UnityEngine;

//////////////////////////////////////////////////////////////////
// [Class for enemy movement, collision detection, and beam firing] 
// Handles movement depending on enemy type, collision detection,
// and continuous firing of different beam types
//////////////////////////////////////////////////////////////////
public class EnemyController : MonoBehaviour
{
    /////////////////////////////////
    // [Variable settings]
    /////////////////////////////////

    // [Constant settings]
    const float BASIC_SPEED = 3;  // Basic enemy speed

    // [Variables set from the Unity Editor]
    [SerializeField] int enemy_Type;   // Enemy type (3 types: 1 or 2 or 3)
    [SerializeField] int enemy_LV;     // Enemy level (1 or 2)

    [SerializeField] int enemy_HP;     // Enemy HP
    [SerializeField] int enemy_Score;  // Score awarded when the enemy is defeated

    [SerializeField] GameObject pre_Beam1;  // Prefab for enemy Beam1
    [SerializeField] GameObject pre_Beam2;  // Prefab for enemy Beam2

    // [Private variables used in this program]
    float speed_X = 0;  // Enemy speed in the X direction
    float speed_Y = 0;  // Enemy speed in the Y direction
    GameObject go_Fighter;  // Fighter game object

    float time_Beam1;  // Firing interval for Beam1
    float time = 0;  // Timer for direction changes of Enemy3
    Vector3 i_Scale;  // Initial scale of the enemy


    /////////////////////////////////
    // [Function settings]
    /////////////////////////////////

    // [Function that runs only once at game start or when the game object is generated]
    // (Unity standard function)
    void Start()
    {
        // Get the Fighter game object in the scene
        go_Fighter = GameObject.Find("SpaceFighter");

        // Get the initial scale of the enemy
        i_Scale = transform.localScale;

        // Beam1 firing interval when enemy level is 1
        if (enemy_LV == 1) time_Beam1 = 0.6f;
        // Beam1 firing interval when enemy level is 2
        else time_Beam1 = 0.4f;

        // Call coroutine function Create_Beam1
        StartCoroutine(Create_Beam1());

        // When enemy_Type is 1
        if (enemy_Type == 1)
        {
            // When the enemy appears on the right side of the screen
            if (transform.position.x >= 0)
                // Set X-direction speed to a random negative value
                speed_X = Random.Range(-1.5f * BASIC_SPEED, -1 * BASIC_SPEED);
            // When the enemy appears on the left side of the screen
            else
                // Set X-direction speed to a random positive value
                speed_X = Random.Range(BASIC_SPEED, 1.5f * BASIC_SPEED);

            // Negative Y-direction speed
            speed_Y = -1 * BASIC_SPEED / 2;
        }

        // When enemy_Type is 2
        else if (enemy_Type == 2)
        {
            // X-direction speed is 0
            speed_X = 0;
            // Negative Y-direction speed
            speed_Y = -1 * BASIC_SPEED;
        }

        // When enemy_Type is 3
        else if (enemy_Type == 3)
        {
            // X-direction speed is 0
            speed_X = 0;
            // Negative Y-direction speed
            speed_Y = -1 * BASIC_SPEED;

            // Call function Create_Beam2
            Create_Beam2();
        }
    }


    // [Function that runs once every frame]
    // (Unity standard function)
    void Update()
    {
        // When the right of the right side of the screen, make the X-speed negative
        if (transform.position.x > 2.4) speed_X = -1 * Mathf.Abs(speed_X);
        // When the left of the left side of the screen, make the X-speed positive
        if (transform.position.x < -2.4) speed_X = Mathf.Abs(speed_X);

        // When the Y coordinate is above the upper side of the screen
        if (transform.position.y > 4.5)
        {
            // Make the Y-direction speed negative
            speed_Y = -1 * Mathf.Abs(speed_Y);
            // Reset the scale to its initial value (sprite facing downward)
            transform.localScale = i_Scale;
        }
        // When the Y coordinate is below the lower side of the screen
        if (transform.position.y < -4.5)
        {
            // Make the Y-direction speed positive
            speed_Y = Mathf.Abs(speed_Y);
            // Change the scale (sprite facing upward)
            transform.localScale = new Vector3(i_Scale.x, -1 * i_Scale.y, 1);
        }

        // When enemy_Type is 3
        if (enemy_Type == 3)
        {
            // Add frame interval time to the timer for Enemy3 direction changes
            time += Time.deltaTime;

            // When the timer for Enemy3 direction changes exceeds 2 seconds
            if (time > 2)
            {
                // Randomly determine the X-direction movement
                switch (Random.Range(1, 4))
                {
                    // If the random value is 1, stop in the X direction
                    case 1:
                        speed_X = 0;
                        break;
                    // If the random value is 2, move right
                    case 2:
                        speed_X = BASIC_SPEED;
                        break;
                    // If the random value is 3, move left
                    case 3:
                        speed_X = -1 * BASIC_SPEED;
                        break;
                }

                // Randomly determine the Y-direction movement
                switch (Random.Range(1, 4))
                {
                    // If the random value is 1, stop in the Y direction
                    case 1:
                        speed_Y = 0;
                        break;
                    // If the random value is 2, move up
                    case 2:
                        speed_Y = BASIC_SPEED;
                        break;
                    // If the random value is 3, move down
                    case 3:
                        speed_Y = -1 * BASIC_SPEED;
                        break;
                }

                // Reset the timer for Enemy3 direction changes to 0
                time = 0;
            }
        }

        // Move the enemy at a constant speed in the specified X and Y directions
        transform.Translate
            (speed_X * Time.deltaTime, speed_Y * Time.deltaTime, 0);
    }


    // [Function called when the enemy’s collider collides with another collider]
    // (Unity standard function)
    private void OnTriggerEnter2D(Collider2D collision)
    {
        // When the tag of the object hit is "Beam1_F" or "Beam2_F"
        if (collision.gameObject.tag == "Beam1_F"
            || collision.gameObject.tag == "Beam2_F")
        {
            // Decrease the enemy’s HP by 1
            enemy_HP--;

            // When the tag of the object hit is "Beam1_F"
            if (collision.gameObject.tag == "Beam1_F")
                // Delete the Fighter’s Beam1 that hit
                Destroy(collision.gameObject);
        }

        // When the enemy’s HP is 0 or less
        if (enemy_HP <= 0)
        {
            // Add the score gained for defeating the enemy to FighterController.score
            go_Fighter.GetComponent<FighterController>().score += enemy_Score;
            // Delete the enemy
            Destroy(this.gameObject);
        }
    }


    // [Coroutine to generate Beam1]
    IEnumerator Create_Beam1()
    {
        Vector3 pos = transform.position;  // Enemy position at the moment of firing

        // When enemy_Type is 1 or 3
        if (enemy_Type == 1 || enemy_Type == 3)
        {
            // Generate Beam1 (pre_Beam1) in the scene (enemy lower-right, downward)
            Instantiate(pre_Beam1, new Vector3(pos.x + 0.3f, pos.y - 0.4f, pos.z),
                Quaternion.identity);
            // Generate Beam1 (pre_Beam1) in the scene (enemy lower-left, downward)
            Instantiate(pre_Beam1, new Vector3(pos.x - 0.3f, pos.y - 0.4f, pos.z),
                Quaternion.identity);
            // Generate Beam1 (pre_Beam1) in the scene (enemy upper-right, upward)
            Instantiate(pre_Beam1, new Vector3(pos.x + 0.3f, pos.y + 0.4f, pos.z),
                Quaternion.Euler(0, 0, 180));
            // Generate Beam1 (pre_Beam1) in the scene (enemy upper-left, upward)
            Instantiate(pre_Beam1, new Vector3(pos.x - 0.3f, pos.y + 0.4f, pos.z),
                Quaternion.Euler(0, 0, 180));
        }

        // When enemy_Type is 2 or 3
        if (enemy_Type == 2 || enemy_Type == 3)
        {
            // Generate Beam1 (pre_Beam1) in the scene (enemy right, rightward)
            Instantiate(pre_Beam1, new Vector3(pos.x + 0.4f, pos.y, pos.z),
                Quaternion.Euler(0, 0, 90));
            // Generate Beam1 (pre_Beam1) in the scene (enemy left, leftward)
            Instantiate(pre_Beam1, new Vector3(pos.x - 0.4f, pos.y, pos.z),
                Quaternion.Euler(0, 0, -90));
        }

        // When enemy level is 2
        if (enemy_LV == 2)
        {
            // Generate Beam1 (pre_Beam1) (enemy lower-right, diagonally right)
            Instantiate(pre_Beam1, new Vector3(pos.x + 0.3f, pos.y - 0.3f, pos.z),
                Quaternion.Euler(0, 0, 30));
            // Generate Beam1 (pre_Beam1) (enemy lower-left, diagonally left)
            Instantiate(pre_Beam1, new Vector3(pos.x - 0.3f, pos.y - 0.3f, pos.z),
                Quaternion.Euler(0, 0, -30));
            // Generate Beam1 (pre_Beam1) (enemy upper-right, diagonally right)
            Instantiate(pre_Beam1, new Vector3(pos.x + 0.3f, pos.y + 0.3f, pos.z),
                Quaternion.Euler(0, 0, 150));
            // Generate Beam1 (pre_Beam1) (enemy upper-left, diagonally left)
            Instantiate(pre_Beam1, new Vector3(pos.x - 0.3f, pos.y + 0.3f, pos.z),
                Quaternion.Euler(0, 0, -150));
        }

        // Wait for the specified number of seconds
        yield return new WaitForSeconds(time_Beam1);

        // If the Fighter still exists, call coroutine function Create_Beam1 again
        if (go_Fighter) StartCoroutine(Create_Beam1());
    }


    // [Function to generate Beam2]
    void Create_Beam2()
    {
        Vector3 pos = transform.position;  // Enemy position at the moment of firing

        // Generate Beam2 (pre_Beam2) in the scene (below the enemy)
        GameObject beamA = Instantiate(pre_Beam2,
            new Vector3(pos.x, pos.y - 1, pos.z), Quaternion.Euler(0, 0, 90));
        // Set the rotation angle that changes the beam’s direction of travel
        beamA.GetComponent<BeamController>().beam_Angle = 200;
        // Set the generated beam’s game object under this enemy as its parent
        beamA.transform.parent = this.transform;

        // Generate Beam2 (pre_Beam2) in the scene (right of the enemy)
        GameObject beamB = Instantiate(pre_Beam2,
            new Vector3(pos.x + 1, pos.y, pos.z), Quaternion.Euler(0, 0, 180));
        // Set the rotation angle that changes the beam’s direction of travel
        beamB.GetComponent<BeamController>().beam_Angle = 200;
        // Set the generated beam’s game object under this enemy as its parent
        beamB.transform.parent = this.transform;

        // Generate Beam2 (pre_Beam2) in the scene (above the enemy)
        GameObject beamC = Instantiate(pre_Beam2,
            new Vector3(pos.x, pos.y + 1, pos.z), Quaternion.Euler(0, 0, -90));
        // Set the rotation angle that changes the beam’s direction of travel
        beamC.GetComponent<BeamController>().beam_Angle = 200;
        // Set the generated beam’s game object under this enemy as its parent
        beamC.transform.parent = this.transform;

        // Generate Beam2 (pre_Beam2) in the scene (left of the enemy)
        GameObject beamD = Instantiate(pre_Beam2,
            new Vector3(pos.x - 1, pos.y, pos.z), Quaternion.Euler(0, 0, 0));
        // Set the rotation angle that changes the beam’s direction of travel
        beamD.GetComponent<BeamController>().beam_Angle = 200;
        // Set the generated beam’s game object under this enemy as its parent
        beamD.transform.parent = this.transform;
    }
}
