﻿using System.Collections;
using UnityEngine;

//////////////////////////////////////////////////////////////////
// [Class for boss movement, collision detection, and beam firing] 
// Handles boss movement in all directions, collision detection,
// and continuous firing of different beam types
//////////////////////////////////////////////////////////////////
public class BossController : MonoBehaviour
{
    /////////////////////////////////
    // [Variable settings]
    /////////////////////////////////

    // [Constant settings]
    const int BOSS_MAXHP = 500;  // Initial boss HP

    // [Variables set from the Unity Editor]
    [SerializeField] GameObject pre_Beam1;  // Prefab for enemy Beam1
    [SerializeField] GameObject pre_Beam3;  // Prefab for enemy Beam3

    [SerializeField] GameController gameController;  // GameController class

    // [Private variables used in this program]
    int boss_HP = BOSS_MAXHP;  // Boss HP
    float speed_X = 0.9f;  // Boss speed in the X direction
    float speed_Y = -0.9f;  // Boss speed in the Y direction


    /////////////////////////////////
    // [Function settings]
    /////////////////////////////////

    // [Function that runs only once at game start or when the game object is generated]
    // (Unity standard function)
    void Start()
    {
        // Call coroutine function Create_Beam1
        StartCoroutine(Create_Beam1());

        // Call coroutine function Create_Beam3
        StartCoroutine(Create_Beam3());
    }


    // [Function that runs once every frame]
    // (Unity standard function)
    void Update()
    {
        // When the Y coordinate is below the center of the screen
        if (transform.position.y < 1)
            // Make the Y-direction speed a positive value
            speed_Y = Mathf.Abs(speed_Y);
        // When the Y coordinate is above upper part of the screen
        else if (transform.position.y > 6)
            // Make the Y-direction speed a negative value
            speed_Y = -1 * Mathf.Abs(speed_Y);

        // When the X coordinate is to left of the left side of the screen
        if (transform.position.x < -2)
            // Make the X-direction speed a positive value
            speed_X = Mathf.Abs(speed_X);
        // When the X coordinate is to right of the right side of the screen
        else if (transform.position.x > 2)
            // Make the X-direction speed a negative value
            speed_X = -1 * Mathf.Abs(speed_X);

        // Move the boss at a constant speed in the specified X and Y directions
        transform.Translate(speed_X * Time.deltaTime, speed_Y * Time.deltaTime, 0);
    }


    // [Function called when the boss’s collider hits another collider]
    // (Unity standard function)
    private void OnTriggerEnter2D(Collider2D collision)
    {
        // When the tag of the object hit is "Beam1_F"
        if (collision.gameObject.tag == "Beam1_F")
        {
            // Decrease the boss’s HP by 1
            boss_HP--;
            // Delete the beam that hit
            Destroy(collision.gameObject);
        }

        // When the tag of the object hit is "Beam2_F"
        else if (collision.gameObject.tag == "Beam2_F")
        {
            // Decrease the boss’s HP by 2
            boss_HP -= 2;
        }

        // When the boss’s current HP is 0 or less
        if (boss_HP <= 0)
        {
            // Call the Game Clear function in the GameController class
            gameController.OnGameClear();
            // Delete the boss
            Destroy(this.gameObject);
        }
    }


    // [Coroutine to generate Beam1]
    IEnumerator Create_Beam1()
    {
        Vector3 pos = transform.position;  // Boss position at the firing timing

        // Generate Beam1 (pre_Beam1) in the scene (front-center of the boss)
        Instantiate(pre_Beam1,
            new Vector3(pos.x, pos.y - 1.5f, pos.z), Quaternion.identity);
        // Generate Beam1 (pre_Beam1) in the scene (front-right of the boss)
        Instantiate(pre_Beam1,
            new Vector3(pos.x + 0.5f, pos.y - 1.2f, pos.z), Quaternion.identity);
        // Generate Beam1 (pre_Beam1) in the scene (front-left of the boss)
        Instantiate(pre_Beam1,
            new Vector3(pos.x - 0.5f, pos.y - 1.2f, pos.z), Quaternion.identity);
        // Generate Beam1 (pre_Beam1) in the scene (rear-right of the boss)
        Instantiate(pre_Beam1,
            new Vector3(pos.x + 0.9f, pos.y + 1.2f, pos.z), Quaternion.identity);
        // Generate Beam1 (pre_Beam1) in the scene (rear-left of the boss)
        Instantiate(pre_Beam1,
            new Vector3(pos.x - 0.9f, pos.y + 1.2f, pos.z), Quaternion.identity);
        // Generate Beam1 (pre_Beam1) in the scene (front-right of the boss)
        Instantiate(pre_Beam1,
            new Vector3(pos.x + 1.7f, pos.y - 0.5f, pos.z), Quaternion.identity);
        // Generate Beam1 (pre_Beam1) in the scene (front-left of the boss)
        Instantiate(pre_Beam1,
            new Vector3(pos.x - 1.7f, pos.y - 0.5f, pos.z), Quaternion.identity);

        // Generate Beam1 (pre_Beam1) in the scene (aimed diagonally down-right)
        Instantiate(pre_Beam1, pos, Quaternion.Euler(0, 0, 15));
        // Generate Beam1 (pre_Beam1) in the scene (aimed diagonally down-left)
        Instantiate(pre_Beam1, pos, Quaternion.Euler(0, 0, -15));

        // Wait for the specified number of seconds
        yield return new WaitForSeconds(0.4f);

        // If the game is not finished, call coroutine function Create_Beam1 again
        if (!gameController.isFinish) StartCoroutine(Create_Beam1());
    }


    // [Coroutine to generate Beam3]
    IEnumerator Create_Beam3()
    {
        // Wait for a random number of seconds
        yield return new WaitForSeconds(Random.Range(3.5f, 5.5f));

        Vector3 pos = transform.position;  // Boss position at the moment of firing

        // Generate Beam3 (pre_Beam3) in the scene (front-center of the boss)
        Instantiate(pre_Beam3,
            new Vector3(pos.x, pos.y - 2.5f, pos.z), Quaternion.identity);
        // Generate Beam3 (pre_Beam3) in the scene (front-right of the boss)
        Instantiate(pre_Beam3,
            new Vector3(pos.x + 3, pos.y - 0.5f, pos.z), Quaternion.identity);
        // Generate Beam3 (pre_Beam3) in the scene (front-left of the boss)
        Instantiate(pre_Beam3,
            new Vector3(pos.x - 3, pos.y - 0.5f, pos.z), Quaternion.identity);

        // If the game is not finished, call coroutine function Create_Beam3 again
        if (!gameController.isFinish) StartCoroutine(Create_Beam3());
    }
}
