﻿using UnityEngine;

//////////////////////////////////////////////////////////////////
// [Class for beam movement and deletion processing]
// Moves and rotates beams, and deletes them when the deletion conditions are met
//////////////////////////////////////////////////////////////////
public class BeamController : MonoBehaviour
{
    /////////////////////////////////
    // [Variable settings]
    /////////////////////////////////

    // [Public variables accessed from other classes]
    public float beam_Angle = 0;  // Angle that changes the beam direction per frame
    // Initial value 0: not curve; The curving beam based on the shooter’s settings

    // [Variables set from the Unity Editor]
    [SerializeField] float beam_Speed;  // Beam speed

    [SerializeField] int beam_Type;  // Beam type (1 or 2 or 3)
    // 1: Straight beam, 2: Rotating beam, 3: Boss beam
    [SerializeField] bool for_Fighter;  // True for beams used by the Fighter

    [SerializeField] SpriteRenderer sRenderer;  // SpriteRenderer


    /////////////////////////////////
    // [Function settings]
    /////////////////////////////////

    // [Function that runs once every frame]
    // (Unity standard function)
    void Update()
    {
        // When the beam type is 1 or 3 (excluding 2)
        if (beam_Type == 1 || beam_Type == 3)
        {
            // When the beam leaves the screen
            if (!sRenderer.isVisible)
            {
                // Delete the beam’s game object
                Destroy(gameObject);
            }
        }

        // Rotate the beam direction by a fixed angle
        transform.Rotate(0, 0, beam_Angle * Time.deltaTime);

        // Move the beam at a constant speed in the local “up” direction
        transform.Translate(0, beam_Speed * Time.deltaTime, 0);
    }


    // [Function called when the beam’s collider hits another collider]
    // (Unity standard function)
    // Used for collision detection between beams
    private void OnTriggerEnter2D(Collider2D collision)
    {
        // In the case of an enemy beam
        if (!for_Fighter)
        {
            // If beam type is 1 and the collided beam tag is "Beam1_F" or "Beam2_F"
            if (beam_Type == 1 &&
                (collision.gameObject.tag == "Beam1_F"
                || collision.gameObject.tag == "Beam2_F"))
            {
                // Delete the beam
                Destroy(this.gameObject);
            }
        }

        // In the case of a Fighter beam
        if (for_Fighter)
        {
            // If beam type is 1 and the collided beam tag is "Beam1_E" or "Beam2_E"
            if (beam_Type == 1 &&
                (collision.gameObject.tag == "Beam1_E"
                || collision.gameObject.tag == "Beam2_E"))
            {
                // Delete the beam
                Destroy(this.gameObject);
            }

            // When the tag of the object hit is "Beam3_E"
            if (collision.gameObject.tag == "Beam3_E")
            {
                // Delete the beam
                Destroy(this.gameObject);
            }
        }
    }
}
