﻿using System.Collections;
using UnityEngine;

//////////////////////////////////////////////////////////////////
// [Class for key-controlled Ninja movement and attack processing]
// Ninja movement by key input and the actions for swinging sword & throwing shuriken
// Handles collision detection when the sword is swung
//////////////////////////////////////////////////////////////////
public class NinjaController : MonoBehaviour
{
    /////////////////////////////////
    // [Variable settings]
    /////////////////////////////////

    // [Variables set from the Unity Editor]
    [SerializeField] GameObject shuriken_R;  // Prefab for right-moving shuriken
    [SerializeField] GameObject shuriken_L;  // Prefab for left-moving shuriken
    [SerializeField] Animator animator;  // Animator component

    // [Private variables used in this program]    
    bool isSwordAction = false;  // Flag whether sword-swinging action is in progress
    bool isShurikenAction = false;  // Flag whether shuriken-throwing is in progress


    /////////////////////////////////
    // [Function settings]
    /////////////////////////////////

    // [Function that runs once every frame]
    // (Unity standard function)
    void Update()
    {
        // When the right arrow key is pressed
        if (Input.GetKey(KeyCode.RightArrow))
        {
            // Face the image to the right
            transform.localScale = new Vector3(0.6f, 0.6f, 1);
            // Set the value of the Animator parameter
            animator.SetInteger("NinjaMotion", 1);
        }

        // When the left arrow key is pressed
        else if (Input.GetKey(KeyCode.LeftArrow))
        {
            // Face the image to the left
            transform.localScale = new Vector3(-0.6f, 0.6f, 1);
            // This game does not allow moving left, so there is no “run left” animation
            animator.SetInteger("NinjaMotion", 0);
        }

        // When the down arrow key is pressed and isShurikenAction is false
        else if (Input.GetKeyDown(KeyCode.DownArrow) && !isShurikenAction)
        {
            // Set the value of the Animator parameter
            animator.SetInteger("NinjaMotion", 2);
            // Call the coroutine function SwordAction
            StartCoroutine(SwordAction());
        }

        // When space key is pressed and both isShurikenAction & isSwordAction are false
        else if (Input.GetKeyDown(KeyCode.Space) && !isShurikenAction && !isSwordAction)
        {
            // Set the value of the Animator parameter
            animator.SetInteger("NinjaMotion", 3);
            // Call the coroutine function ThrowShuriken
            StartCoroutine(ThrowShuriken());
        }

        // In all other cases
        else
        {
            // Set the value of the Animator parameter
            animator.SetInteger("NinjaMotion", 0);
        }
    }


    // [Function called when the ninja’s outer collider contact with another collider]
    // (Unity standard function)
    private void OnTriggerStay2D(Collider2D collision)
    {
        // When the isSwordAction flag is true (the sword is being swung)
        if (isSwordAction)
        {
            // Get the position of the object that was hit
            float colPosX = collision.transform.position.x;
            // If facing right and the hit target is slightly to the right, or
            // if facing left and the hit target is slightly to the left
            if ((transform.localScale.x > 0 && colPosX > transform.position.x + 0.5)
                || (transform.localScale.x < 0 && colPosX < transform.position.x - 0.5))
            {
                // If the tag of the object that was hit is “Enemy_Body”
                if (collision.gameObject.CompareTag("Enemy_Body"))
                {
                    // Delete the parent game object of the object that was hit (enemy)
                    Destroy(collision.gameObject.transform.parent.gameObject);
                }

                // If the tag of the object that was hit is “Sword”
                else if (collision.gameObject.CompareTag("Sword"))
                {
                    // Delete the object that was hit (enemy weapon)
                    Destroy(collision.gameObject);
                }
            }
        }
    }


    // [Coroutine that performs the shuriken-throwing action]
    IEnumerator ThrowShuriken()
    {
        // Set the isShurikenAction flag to true
        isShurikenAction = true;

        // Wait for the specified number of seconds
        yield return new WaitForSeconds(0.3f);

        // Since thrown shuriken is generated based on the ninja’s current position
        Vector3 pos = transform.position;
        // If the ninja is facing right
        if (transform.localScale.x > 0)
        {
            // Generate a right-moving shuriken at the specified position in the scene
            Instantiate(shuriken_R,
                new Vector3(pos.x + 1.5f, pos.y + 2.0f, 0), Quaternion.identity);
        }
        // If the ninja is facing left
        else
        {
            // Generate a left-moving shuriken at the specified position in the scene
            Instantiate(shuriken_L,
                new Vector3(pos.x - 1.5f, pos.y + 2.0f, 0), Quaternion.identity);
        }

        // Wait for the specified number of seconds
        yield return new WaitForSeconds(0.3f);

        // Set the isShurikenAction flag to false
        isShurikenAction = false;
    }


    // [Coroutine that performs the sword-swinging action]
    IEnumerator SwordAction()
    {
        // Set the isSwordAction flag to true
        isSwordAction = true;

        // Wait for the specified number of seconds
        yield return new WaitForSeconds(0.5f);

        // Set the isSwordAction flag to false
        isSwordAction = false;
    }
}
