﻿using System.Collections;
using UnityEngine;

//////////////////////////////////////////////////////////////////
// [Class used for automatic movement and automatic attack for green and red enemy]
// Automatic enemy movement and random attack actions (swinging & throwing sword/axe)
//////////////////////////////////////////////////////////////////
public class EnemyController : MonoBehaviour
{
    /////////////////////////////////
    // [Variable settings]
    /////////////////////////////////

    // [Constant settings]
    const float ENEMY_SPEED_X = 0.02f;  // Maximum enemy movement speed
    const float TIME_SWORD = 3.0f;  // Maximum time until the next attack action

    // [Public variables accessed from other classes]
    public int direction = -1;  // Direction of the enemy (1: right, -1: left)

    // [Variables set from the Unity Editor]
    [SerializeField] int enemyType = 1;  // Type of enemy (1: green enemy, 2: red enemy)
    [SerializeField] Animator animator;  // Animator component
    [SerializeField] GameObject pre_Sword_R;  // Prefab of enemy’s thrown weapon (right)
    [SerializeField] GameObject pre_Sword_L;  // Prefab of enemy’s thrown weapon (left)

    // [Private variables used in this program]    
    GameController cl_GameController; // GameController class
    float speed_X;  // Enemy movement speed
    bool isSwordAction = false;  // Flag whether sword-swinging action is in progress


    /////////////////////////////////
    // [Function settings]
    /////////////////////////////////

    // [Function that runs only once at game start or when the game object is generated]
    // (Unity standard function)
    void Start()
    {
        // Get the GameController class from “GameController” game object in the scene
        cl_GameController =
            GameObject.Find("GameController").GetComponent<GameController>();

        // When enemyType is 1 (green enemy)
        if (enemyType == 1)
        {
            // Get a random value within the specified range as the movement speed
            speed_X = Random.Range(ENEMY_SPEED_X - 0.01f, ENEMY_SPEED_X);
        }
        // When enemyType is 2 (red enemy), use a fixed movement speed
        else speed_X = ENEMY_SPEED_X - 0.015f;

        // Call the coroutine function SwordAction
        StartCoroutine(SwordAction());
    }


    // [Function that runs once every frame]
    // (Unity standard function)
    void Update()
    {
        // If GameController.isGameFinish is true
        if (cl_GameController.isGameFinish) return;

        // Define a speed used to adjust relative speed when the ninja is moving right
        float addSpeed = 0;

        // When the right arrow key is pressed
        if (Input.GetKey(KeyCode.RightArrow))
        {
            // Since the ninja is moving to the right, adjust the relative speed
            addSpeed = 0.02f;
        }

        // Move the enemy in the specified direction at the specified speed
        transform.Translate(direction * speed_X - addSpeed, 0, 0);
    }


    // [Function called when the enemy’s outer collider contact with another collider]
    // (Unity standard function)
    private void OnTriggerStay2D(Collider2D collision)
    {
        // When the tag of the object hit is “Ninja_Body” and isSwordAction flag is true
        if (collision.gameObject.CompareTag("Ninja_Body") && isSwordAction)
        {
            // Get the position of the object that was hit
            float colPosX = collision.transform.position.x;
            // If facing right and the hit target is to the right of self, or
            // if facing left and the hit target is to the left of self
            if ((transform.localScale.x > 0 && colPosX > transform.position.x)
                || (transform.localScale.x < 0 && colPosX < transform.position.x))
            {
                // Delete the parent game object of the object that was hit (ninja)
                Destroy(collision.gameObject.transform.parent.gameObject);

                // Call the GameOver function of the GameController class
                cl_GameController.GameOver();
            }
        }
    }


    // [Coroutine that performs the sword-swinging action]
    IEnumerator SwordAction()
    {
        // Wait for a random number of seconds within the specified range
        yield return new WaitForSeconds(Random.Range(TIME_SWORD - 1.5f, TIME_SWORD));

        // If the random value between 0 and 1 is greater than 0.7 (swing the sword/axe)
        if (Random.value > 0.7f)
        {
            // Set the isSwordAction flag to true
            isSwordAction = true;
            // Set the value of the Animator parameter
            animator.SetBool("EnemySword", true);
        }
        // If the random value between 0 and 1 is 0.7 or less (throw the sword/axe)
        else
        {
            // Set the value of the Animator parameter
            animator.SetBool("EnemyThrow", true);
            // Call the coroutine function ThrowSword
            StartCoroutine(ThrowSword());
        }

        // Wait for the specified number of seconds
        yield return new WaitForSeconds(0.45f);

        // Set the value of the Animator parameter
        animator.SetBool("EnemySword", false);
        // Set the value of the Animator parameter
        animator.SetBool("EnemyThrow", false);
        // Set the isSwordAction flag to false
        isSwordAction = false;

        // If isGameFinish is false, call the coroutine function SwordAction
        if (!cl_GameController.isGameFinish) StartCoroutine(SwordAction());
    }


    // [Coroutine that performs the sword/axe throwing action]
    IEnumerator ThrowSword()
    {
        // Wait for the specified number of seconds
        yield return new WaitForSeconds(0.4f);

        // Since thrown sword/axe is generated based on the enemy’s current position
        Vector3 pos = transform.position;
        // If the enemy is facing right
        if (transform.localScale.x > 0)
        {
            // Generate a right-facing sword/axe at the specified position in the scene
            Instantiate(pre_Sword_R,
                new Vector3(pos.x + 1.7f, pos.y + 2f, 0), Quaternion.identity);
        }
        // If the enemy is facing left
        else
        {
            // Generate a left-facing sword/axe at the specified position in the scene
            Instantiate(pre_Sword_L,
                new Vector3(pos.x - 1.7f, pos.y + 2f, 0), Quaternion.identity);
        }
    }
}
