﻿using System.Collections;
using UnityEngine;

using UnityEngine.SceneManagement;  // Used for switching scenes
using UnityEngine.UI;  // Used for UI control processing

//////////////////////////////////////////////////////////////////
// [Class for automatic enemy generation and for Game Over / Game Clear / Restart] 
// Automatic enemy generation processing according to elapsed game time,
// restart processing, and Game Over / Game Clear processing
//////////////////////////////////////////////////////////////////
public class GameController : MonoBehaviour
{
    /////////////////////////////////
    // [Variable settings]
    /////////////////////////////////

    // [Constant settings]
    const float BOSS_START_TIME = 90;  // Time until the boss appears
    const float ENEMY_L2_TIME = 45;  // Time until enemy spawn frequency is increased
    const float TIME_ENEMY1_L1 = 1.5f;  // Enemy1 spawn time at low frequency
    const float TIME_ENEMY1_L2 = 0.7f;  // Enemy1 spawn time at high frequency
    // Enemy2 uses twice Enemy1’s spawn time (Enemy2 appears at half rate of Enemy1)

    // [Public variables accessed from other classes]
    public bool isGameFinish = false;  // Game end flag
    public float time = 0;  // Elapsed game time
    // Used for enemy spawn frequency, boss appearance, and Fighter transforming

    // [Variables set from Unity Editor]
    [SerializeField] GameObject go_GameStatus;
    // Game object of the text UI for displaying Game Over and Game Clear
    [SerializeField] GameObject prefab_Enemy1;  // Prefab of enemy Enemy1
    [SerializeField] GameObject prefab_Enemy2;  // Prefab of enemy Enemy2
    [SerializeField] GameObject go_Enemy_Boss;  // Game object of the enemy boss

    // [Private variables used in this program]
    float time_Max_Enemy1;  // Maximum time until Enemy1 is generated
    float time_Max_Enemy2;  // Maximum time until Enemy2 is generated
    bool mode_Boss = false;  // Boss appearance mode (initial value is false)


    /////////////////////////////////
    // [Function settings]
    /////////////////////////////////

    // [Function that runs only once at game start or when the game object is generated]
    // (Unity standard function)
    void Start()
    {
        // Call coroutine function Create_Enemy1
        StartCoroutine(Create_Enemy1());
        // Call coroutine function Create_Enemy2
        StartCoroutine(Create_Enemy2());
    }


    // [Function that runs once every frame]
    // (Unity standard function)
    void Update()
    {
        // Calculate elapsed game time by adding elapsed time per frame
        time += Time.deltaTime;

        // If elapsed game time is less than the time to increase enemy spawn frequency
        if (time < ENEMY_L2_TIME)
        {
            // Set Enemy1’s spawn time to the low-frequency time
            time_Max_Enemy1 = TIME_ENEMY1_L1;
            // Set Enemy2’s spawn time to twice Enemy1’s spawn time
            time_Max_Enemy2 = TIME_ENEMY1_L1 * 2;
        }
        // If elapsed game time is less than the boss spawn time
        else if (time < BOSS_START_TIME)
        {
            // Set Enemy1’s spawn time to the high-frequency time
            time_Max_Enemy1 = TIME_ENEMY1_L2;
            // Set Enemy2’s spawn time to twice Enemy1’s spawn time
            time_Max_Enemy2 = TIME_ENEMY1_L2 * 2;
        }
        // Otherwise (when the boss spawn time has been exceeded)
        else
        {
            // If the boss appearance mode is false
            if (!mode_Boss)
            {
                // Set the boss appearance mode to true
                mode_Boss = true;
                // Make the boss game object appear
                go_Enemy_Boss.SetActive(true);
            }
        }
    }


    // [Coroutine to generate Enemy1]
    IEnumerator Create_Enemy1()
    {
        // Randomly set the Y coordinate at which Enemy1 appears
        float pos_Spoon_Y = Random.Range(-4.0f, 4.0f);
        // Obtain a random time within the specified range to generate the next Enemy1
        float time_Wait = Random.Range(time_Max_Enemy1 - 0.5f, time_Max_Enemy1);

        // Generate Enemy1 (prefab_Enemy1) at the specified position in the scene
        Instantiate(prefab_Enemy1, new Vector3(10, pos_Spoon_Y, 0), Quaternion.identity);

        // Wait for the specified seconds
        yield return new WaitForSeconds(time_Wait);

        // If not in boss appearance mode, call the coroutine function Create_Enemy1
        if (!mode_Boss) StartCoroutine(Create_Enemy1());
    }


    // [Coroutine to generate Enemy2]
    IEnumerator Create_Enemy2()
    {
        // Randomly set the Y coordinate at which Enemy2 appears
        float pos_Spoon_Y = Random.Range(-4.0f, 4.0f);
        // Obtain a random time within the specified range to generate the next Enemy2
        float time_Wait = Random.Range(time_Max_Enemy2 - 0.5f, time_Max_Enemy2);

        // Generate Enemy2 (prefab_Enemy2) at the specified position in the scene
        Instantiate(prefab_Enemy2, new Vector3(10, pos_Spoon_Y, 0), Quaternion.identity);

        // Wait for the specified seconds
        yield return new WaitForSeconds(time_Wait);

        // If not in boss appearance mode, call the coroutine function Create_Enemy2
        if (!mode_Boss) StartCoroutine(Create_Enemy2());
    }


    // [Game Over processing function]
    public void GameOver()
    {
        // If the game end flag is already true, do not execute subsequent processing
        // (So that Game Over is not executed after Game Clear)
        if (isGameFinish) return;

        // Set the game end flag to true
        isGameFinish = true;
        // Enable the game object of the text UI for displaying Game Over
        go_GameStatus.SetActive(true);
        // Display "Game Over" on the text UI
        go_GameStatus.GetComponent<Text>().text = "Game Over";
    }


    // [Game Clear processing function]
    public void GameClear()
    {
        // If the game end flag is already true, do not execute subsequent processing
        // (So that Game Clear is not executed after Game Over)
        if (isGameFinish) return;

        // Set the game end flag to true
        isGameFinish = true;
        // Enable the game object of the text UI for displaying Game Clear
        go_GameStatus.SetActive(true);
        // Display "Game Clear" on the text UI
        go_GameStatus.GetComponent<Text>().text = "Game Clear";
    }


    // [Restart processing function called from the Restart button]
    public void OnClick_Restart()
    {
        // Call the scene "GameMain"
        SceneManager.LoadScene("GameMain");
    }
}
