﻿using System.Collections;
using UnityEngine;

using UnityEngine.UI;  // Used for UI control processing

//////////////////////////////////////////////////////////////////
// [Class for Fighter movement, collision detection, and beam firing process] 
// Movement by mouse control, beam firing processing according to beam type and Fighter mode,
// transformation processing according to elapsed time, collision detection processing,
// and control of the energy and HP gauge UI
//////////////////////////////////////////////////////////////////
public class FighterController : MonoBehaviour
{
    /////////////////////////////////
    // [Variable settings]
    /////////////////////////////////

    // [Constant settings]    
    const float FIGHTER_MAX_HP = 10;  // Fighter’s initial HP
    const float SHOOT_MAX_POWER = 10;  // Fireable energy of the spread beam cannon
    const float POWER_CHARGE = 1.5f;  // Increase amount of energy for the spread beam
    const float SHOOT_NUM = 15;  // Number of spreads for the Fighter’s spread beam
    const float FIGHTER_CHANGE_TIME = 45;  // Time until the Fighter transforms

    // [Variables set from Unity Editor]
    [SerializeField] GameObject prefab_Beam1;  // Fighter’s Beam 1 Prefab (Beam1)
    [SerializeField] GameObject prefab_Beam2;  // Fighter’s Beam 2 Prefab (Beam2)

    [SerializeField] GameController cl_GameController;  // GameController class

    [SerializeField] Image hpGauge;  // Gauge UI image that displays the current Fighter HP 
    [SerializeField] Image powerGauge;  // Gauge UI image for the current spread beam energy 
    [SerializeField] Sprite sprite_FighterL2;  // Fighter’s transformed mode Sprite image

    [SerializeField] AudioClip se_BigExplosion;  // Fighter explosion sound effect
    [SerializeField] AudioClip se_Shot;  // Beam cannon firing sound
    [SerializeField] AudioSource audioSource;  // AudioSource component

    // [Private variables used in this program]
    float fighter_HP = FIGHTER_MAX_HP;  // Current HP of the Fighter
    int fighter_Mode = 1;  // Current Fighter mode (1: initial mode, 2: transformed mode)
    float shoot_Power = 0;  // Amount of energy for the spread beam

    Vector3 startPos;  // Initial on-screen position of mouse movement in one frame


    /////////////////////////////////
    // [Function settings]
    /////////////////////////////////

    // [Function that runs only once at game start or when the game object is generated]
    // (Unity standard function)
    void Start()
    {
        // Reflect the Fighter’s initial HP value in the HP gauge UI image
        hpGauge.fillAmount = fighter_HP / FIGHTER_MAX_HP;

        // Call coroutine function Create_Beam1
        StartCoroutine(Create_Beam1());
    }


    // [Function that runs once every frame]
    // (Unity standard function)
    void Update()
    {
        // When the GameController’s elapsed game time exceeds the Fighter transform time
        if (fighter_Mode == 1 && cl_GameController.time >= FIGHTER_CHANGE_TIME)
        {
            // Set the Fighter mode to transformed mode
            fighter_Mode = 2;
            // Change the Fighter’s Sprite to the transformed mode Sprite
            this.gameObject.GetComponent<SpriteRenderer>().sprite = sprite_FighterL2;
        }

        // When the left mouse button is pressed
        if (Input.GetMouseButtonDown(0))
        {
            // Get the current on-screen coordinates of the mouse cursor
            startPos = Camera.main.ScreenToWorldPoint(Input.mousePosition);
        }

        // When the left mouse button is being held down
        if (Input.GetMouseButton(0))
        {
            // Get the current on-screen coordinates of the mouse cursor (end of one frame)
            Vector3 endPos = Camera.main.ScreenToWorldPoint(Input.mousePosition);

            // Mouse movement vector for one frame (mouse end position − initial position)
            Vector3 difPos = endPos - startPos;

            // If (the Fighter’s current position + the mouse movement) is within the screen
            if (8 > transform.position.x + difPos.x &&
                    transform.position.x + difPos.x > -8 &&
                4.5 > transform.position.y + difPos.y &&
                    transform.position.y + difPos.y > -4.5)
            {
                // Move from the Fighter’s current position by the mouse movement vector
                transform.Translate(difPos);
            }

            // Obtain the initial mouse position to calculate the vector in the next frame
            startPos = Camera.main.ScreenToWorldPoint(Input.mousePosition);
        }

        // When the right mouse button is pressed,
        // and the current energy for the spread beam is greater than the fireable energy
        if (Input.GetMouseButtonDown(1) && shoot_Power >= SHOOT_MAX_POWER)
        {
            // Call function Create_Beam2
            Crate_Beam2();
        }

        // Add the increase amount for one frame time to the energy for the spread beam
        shoot_Power += Time.deltaTime * POWER_CHARGE;
        // Display the gauge UI in proportion to the current energy for the spread beam
        powerGauge.fillAmount = shoot_Power / SHOOT_MAX_POWER;
    }


    // [Function called when the Fighter’s collider collides with another collider]
    // (Unity standard function)
    private void OnTriggerEnter2D(Collider2D collision)
    {
        // If the tag of the collided object is "Beam_Enemy"
        if (collision.gameObject.tag == "Beam_Enemy")
        {
            // Decrease the Fighter’s HP by 1
            fighter_HP--;

            // If the Fighter’s HP is 0 or more
            if (fighter_HP >= 0)
            {
                // Display the HP gauge UI image in proportion to the current Fighter HP
                hpGauge.fillAmount = fighter_HP / FIGHTER_MAX_HP;
            }
            else
            {
                // Play the Fighter explosion sound effect once
                audioSource.PlayOneShot(se_BigExplosion);

                // Call the Game Over function of the GameController class
                cl_GameController.GameOver();
                // Delete the Fighter
                Destroy(gameObject);
            }
        }
    }


    // [Coroutine to generate Beam1]
    IEnumerator Create_Beam1()
    {
        // Play the beam cannon firing sound once
        audioSource.PlayOneShot(se_Shot);

        // To be fired based on the Fighter’s current position, get the current position
        Vector3 pos = transform.position;

        // Generate a beam (prefab_Beam1) in the scene
        Instantiate(prefab_Beam1,
            new Vector3(pos.x + 0.8f, pos.y - 0.3f, pos.z), Quaternion.identity);

        // If the Fighter mode is transformed mode
        if (fighter_Mode == 2)
        {
            // Generate a beam (prefab_Beam1) in the scene (beam_A)
            GameObject beam_A = Instantiate(prefab_Beam1);
            // Move beam_A to the specified position
            beam_A.transform.position = new Vector3(pos.x + 0.4f, pos.y + 0.4f, pos.z);
            // Change the BeamController class’s beam_Angle variable value of beam_A
            beam_A.GetComponent<BeamController>().beam_Angle = -20.0f;
            // Change the angle of beam_A to the specified angle
            beam_A.transform.localEulerAngles = new Vector3(0, 0, 25);

            // Generate a beam (prefab_Beam1) in the scene (beam_B)
            GameObject beam_B = Instantiate(prefab_Beam1);
            // Move beam_B to the specified position
            beam_B.transform.position = new Vector3(pos.x + 0.4f, pos.y - 0.45f, pos.z);
            // Change the BeamController class’s beam_Angle variable value of beam_B
            beam_B.GetComponent<BeamController>().beam_Angle = 20.0f;
            // Change the angle of beam_B to the specified angle
            beam_B.transform.localEulerAngles = new Vector3(0, 0, -30);
        }

        // Wait for the specified seconds
        yield return new WaitForSeconds(0.5f);

        // If GameController isGameFinish is false, call the coroutine function Create_Beam1
        if (!cl_GameController.isGameFinish) StartCoroutine(Create_Beam1());
    }


    // [Function to generate Beam2]
    void Crate_Beam2()
    {
        // To be fired based on the Fighter’s current position, get the current position
        Vector3 pos = transform.position;

        // switch statement based on Fighter mode
        switch (fighter_Mode)
        {
            // If Fighter mode is 1 (initial mode)
            case 1:
                // Repeat for as many times as the number of spreads of the spread beam
                for (int i = 1; i <= SHOOT_NUM; i++)
                {
                    // Generate a beam (prefab_Beam2) at the Fighter’s position (beam_A)
                    GameObject beam_A =
                        Instantiate(prefab_Beam2, pos, Quaternion.identity);
                    // Rotate the beam by 360° / number of spreads of the spread beam
                    beam_A.transform.localEulerAngles = new Vector3(0, 0, i * 360 / SHOOT_NUM);
                }

                break;

            // If Fighter mode is 2 (transformed mode)
            case 2:
                // Repeat for as many times as the number of spreads of the spread beam
                for (int i = 1; i <= SHOOT_NUM; i++)
                {
                    // Generate a beam (prefab_Beam2) in the scene (beam_A)
                    GameObject beam_A = Instantiate(prefab_Beam2);
                    // Move beam_A to the specified position
                    beam_A.transform.position = new Vector3(pos.x + 1.3f, pos.y + 0.7f, 0);
                    // Rotate the beam by 360° / number of spreads of the spread beam
                    beam_A.transform.localEulerAngles = new Vector3(0, 0, i * 360 / SHOOT_NUM);

                    // Generate a beam (prefab_Beam2) in the scene (beam_B)
                    GameObject beam_B = Instantiate(prefab_Beam2);
                    // Move beam_B to the specified position
                    beam_B.transform.position = new Vector3(pos.x - 1.3f, pos.y + 0.7f, 0);
                    // Rotate the beam by 360° / number of spreads of the spread beam
                    beam_B.transform.localEulerAngles = new Vector3(0, 0, i * 360 / SHOOT_NUM);

                    // Generate a beam (prefab_Beam2) in the scene (beam_C)
                    GameObject beam_C = Instantiate(prefab_Beam2);
                    // Move beam_C to the specified position
                    beam_C.transform.position = new Vector3(pos.x + 1.3f, pos.y - 0.7f, 0);
                    // Rotate the beam by 360° / number of spreads of the spread beam
                    beam_C.transform.localEulerAngles = new Vector3(0, 0, i * 360 / SHOOT_NUM);

                    // Generate a beam (prefab_Beam2) in the scene (beam_D)
                    GameObject beam_D = Instantiate(prefab_Beam2);
                    // Move beam_D to the specified position
                    beam_D.transform.position = new Vector3(pos.x - 1.3f, pos.y - 0.7f, 0);
                    // Rotate the beam by 360° / number of spreads of the spread beam
                    beam_D.transform.localEulerAngles = new Vector3(0, 0, i * 360 / SHOOT_NUM);
                }

                break;
        }

        // Initialize the current energy amount for the spread beam to 0
        shoot_Power = 0;
    }
}
