﻿using System.Collections;
using UnityEngine;

//////////////////////////////////////////////////////////////////
// [Class for enemy movement, collision detection, and beam firing process] 
// Movement according to the type of enemy, collision detection process, 
// and continuous firing process depending on the type of beam
//////////////////////////////////////////////////////////////////
public class EnemyController : MonoBehaviour
{
    /////////////////////////////////
    // [Variable settings]
    /////////////////////////////////

    // [Constant settings]
    const float SHOOT_NUM = 10;  // Number of spreads for enemy’s spread beam

    // [Variables set from Unity Editor]
    [SerializeField] int enemy_Type = 1;  // Two types of enemies (enemy_Type: 1 or 2)
    [SerializeField] float speed_Y = 0;  // Enemy’s speed in Y direction (0: horizontal)
    [SerializeField] int enemy_HP = 1;  // Enemy HP

    [SerializeField] GameObject prefab_Beam1;  // Enemy beam 1 Prefab (Beam1)
    [SerializeField] GameObject prefab_Beam2;  // Enemy beam 2 Prefab (Beam2)
    [SerializeField] AudioClip se_SmallExplosion;  // Enemy explosion sound effect

    // [Private variables used in this program]
    float speed_X;  // Enemy’s speed in X direction

    GameObject go_Fighter;  // Fighter’s game object
    AudioSource audioSource;  // AudioSource component


    /////////////////////////////////
    // [Function settings]
    /////////////////////////////////

    // [Function that runs only once at game start or when the game object is generated]
    // (Unity standard function)
    void Start()
    {
        // Get the Fighter’s game object in the scene
        go_Fighter = GameObject.Find("Fighter");
        // Get the AudioSource component of the GameController game object in the scene
        audioSource = GameObject.Find("GameController").GetComponent<AudioSource>();

        // Initial settings depending on enemy_Type
        switch (enemy_Type)
        {
            // In case of enemy_Type:1
            case 1:
                // Get a random speed in the X direction
                speed_X = Random.Range(-3.0f, -2.0f);
                break;

            // In case of enemy_Type:2
            case 2:
                // Get a random speed in the X direction
                speed_X = Random.Range(-1.5f, -1.0f);
                break;
        }

        // Call coroutine function Create_Beam1
        StartCoroutine(Create_Beam1());

        // In case of enemy_Type:2
        if (enemy_Type == 2)
            // Call coroutine function Create_Beam2
            StartCoroutine(Create_Beam2());
    }


    // [Function that runs once every frame]
    // (Unity standard function)
    void Update()
    {
        // In case of enemy_Type:2
        if (enemy_Type == 2)
        {
            // If Y coordinate is below a certain position (bottom of the screen)
            if (transform.position.y < -4.2)
                // Set Y direction speed to a positive value
                speed_Y = Mathf.Abs(speed_Y);
            // If Y coordinate is above a certain position (top of the screen)
            else if (transform.position.y > 4.2)
                // Set Y direction speed to a negative value
                speed_Y = -1 * Mathf.Abs(speed_Y);
        }

        // Move the enemy at a constant speed in the specified X and Y directions
        transform.Translate(speed_X * Time.deltaTime, speed_Y * Time.deltaTime, 0);
    }


    // [Function called when the enemy’s collider collides with another collider]
    // (Unity standard function)
    private void OnTriggerEnter2D(Collider2D collision)
    {
        // If the tag of the collided object is "Beam_Fighter"
        if (collision.gameObject.tag == "Beam_Fighter")
        {
            // Decrease enemy HP by 1
            enemy_HP--;

            // If enemy HP becomes 0 or less
            if (enemy_HP <= 0)
            {
                // Play enemy explosion sound effect once
                audioSource.PlayOneShot(se_SmallExplosion);

                // Delete the enemy
                Destroy(gameObject);
            }
        }

        // If the tag of the collided object is "Border"
        if (collision.gameObject.tag == "Border")
        {
            // Delete the enemy
            Destroy(gameObject);
        }
    }


    // [Coroutine to generate Beam1]
    IEnumerator Create_Beam1()
    {
        // If Fighter exists in the scene
        if (go_Fighter)
        {
            // Generate a beam (prefab_Beam1) in the scene
            GameObject beam =
                Instantiate(prefab_Beam1, transform.position, Quaternion.identity);

            // Get the direction of Fighter from the generated beam
            Vector3 dir = go_Fighter.transform.position - beam.transform.position;
            // Point the generated beam toward Fighter
            beam.transform.rotation = Quaternion.FromToRotation(Vector3.right, dir);

            // Wait for the specified seconds
            yield return new WaitForSeconds(1.0f);

            // Call coroutine function Create_Beam1
            StartCoroutine(Create_Beam1());
        }
    }


    // [Coroutine to generate Beam2]
    IEnumerator Create_Beam2()
    {
        // If Fighter exists in the scene
        if (go_Fighter)
        {
            // Repeat as many times as the number of spreads of the spread beam
            for (int i = 1; i <= SHOOT_NUM; i++)
            {
                // Generate a beam (prefab_Beam2) in the scene
                GameObject beam =
                    Instantiate(prefab_Beam2, transform.position, Quaternion.identity);

                // Rotate the beam by 360° / number of spreads of the spread beam
                beam.transform.localEulerAngles = new Vector3(0, 0, i * 360 / SHOOT_NUM);
            }

            // Wait for the specified seconds
            yield return new WaitForSeconds(3.0f);

            // Call coroutine function Create_Beam2
            StartCoroutine(Create_Beam2());
        }
    }
}
