﻿using System.Collections;
using UnityEngine;

//////////////////////////////////////////////////////////////////
// [Class for boss movement, collision detection, and beam firing process] 
// Boss movement up, down, left, and right, collision detection process, 
// and continuous firing process depending on the type of beam
//////////////////////////////////////////////////////////////////
public class BossController : MonoBehaviour
{
    /////////////////////////////////
    // [Variable settings]
    /////////////////////////////////

    // [Constant settings]
    const int BOSS_MAX_HP = 99;  // Boss initial HP
    const float SHOOT_NUM = 12;  // Number of spreads for the boss’s spread beam

    // [Variables set from the Unity Editor]
    [SerializeField] GameObject prefab_Beam1;  // Enemy beam 1 Prefab (Beam1)
    [SerializeField] GameObject prefab_Beam2;  // Enemy beam 2 Prefab (Beam2)
    [SerializeField] GameObject go_Fighter;  // Fighter’s game object
    [SerializeField] GameController cl_GameController;  // GameController class

    [SerializeField] AudioClip se_BigExplosion;  // Boss explosion sound effect
    [SerializeField] AudioSource audioSource;  // AudioSource component

    // [Private variables used in this program]    
    int enemy_HP = BOSS_MAX_HP;  // Boss HP
    float speed_X = -1;  // Boss speed in X direction
    float speed_Y = 1;  // Boss speed in Y direction


    /////////////////////////////////
    // [Function settings]
    /////////////////////////////////

    // [Function that runs only once at game start or when the game object is generated]
    // (Unity standard function)
    void Start()
    {
        // Call coroutine function Create_Beam1
        StartCoroutine(Create_Beam1());
        // Call coroutine function Create_Beam2
        StartCoroutine(Create_Beam2());
    }


    // [Function that runs once every frame]
    // (Unity standard function)
    void Update()
    {
        // If Y coordinate is below a certain position (bottom of the screen)
        if (transform.position.y < -2.5)
            // Set Y direction speed to a positive value
            speed_Y = Mathf.Abs(speed_Y);
        // If Y coordinate is above a certain position (top of the screen)
        else if (transform.position.y > 2.5)
            // Set Y direction speed to a negative value
            speed_Y = -1 * Mathf.Abs(speed_Y);

        // If X coordinate is left of a certain position (left side of the screen)
        if (transform.position.x < -5)
            // Set X direction speed to a positive value
            speed_X = Mathf.Abs(speed_X);
        // If X coordinate is right of a certain position (right side of the screen)
        else if (transform.position.x > 5)
            // Set X direction speed to a negative value
            speed_X = -1 * Mathf.Abs(speed_X);

        // Move the boss at a constant speed in the specified X and Y directions
        transform.Translate(speed_X * Time.deltaTime, speed_Y * Time.deltaTime, 0);
    }


    // [Function called when the boss’s collider collides with another collider]
    // (Unity standard function)
    private void OnTriggerEnter2D(Collider2D collision)
    {
        // If the tag of the collided object is "Beam_Fighter"
        if (collision.gameObject.tag == "Beam_Fighter")
        {
            // Decrease boss HP by 1
            enemy_HP--;

            // If boss HP becomes 0 or less
            if (enemy_HP <= 0)
            {
                // Play boss explosion sound effect once
                audioSource.PlayOneShot(se_BigExplosion);

                // Call the game clear function of the GameController class
                cl_GameController.GameClear();
                // Delete the boss
                Destroy(gameObject);
            }
        }
    }


    // [Coroutine to generate Beam1]
    IEnumerator Create_Beam1()
    {
        // If Fighter exists in the scene
        if (go_Fighter)
        {
            Vector3 pos = transform.position;  // Boss’s coordinates at the firing timing

            // Generate beam (prefab_Beam1) in the scene (beam_A)
            GameObject beam_A = Instantiate(prefab_Beam1);
            // Move beam_A to the specified position
            beam_A.transform.position = new Vector3(pos.x - 3f, pos.y - 1f, 0);
            // Get the direction of Fighter from the generated beam
            Vector3 dir = go_Fighter.transform.position - beam_A.transform.position;
            // Point the generated beam toward Fighter
            beam_A.transform.rotation = Quaternion.FromToRotation(Vector3.right, dir);

            // Generate beam (prefab_Beam1) in the scene (beam_B)
            GameObject beam_B = Instantiate(prefab_Beam1);
            // Move beam_B to the specified position
            beam_B.transform.position = new Vector3(pos.x - 2f, pos.y + 1f, 0);
            // Get the direction of Fighter from the generated beam
            dir = go_Fighter.transform.position - beam_B.transform.position;
            // Point the generated beam toward Fighter
            beam_B.transform.rotation = Quaternion.FromToRotation(Vector3.right, dir);

            // Generate beam (prefab_Beam1) in the scene (beam_C)
            GameObject beam_C = Instantiate(prefab_Beam1);
            // Move beam_C to the specified position
            beam_C.transform.position = new Vector3(pos.x - 1f, pos.y + 2f, 0);
            // Get the direction of Fighter from the generated beam
            dir = go_Fighter.transform.position - beam_C.transform.position;
            // Point the generated beam toward Fighter
            beam_C.transform.rotation = Quaternion.FromToRotation(Vector3.right, dir);

            // Wait for the specified seconds
            yield return new WaitForSeconds(1.0f);

            // Call coroutine function Create_Beam1
            StartCoroutine(Create_Beam1());
        }
    }


    // [Coroutine to generate Beam2]
    IEnumerator Create_Beam2()
    {
        // If Fighter exists in the scene
        if (go_Fighter)
        {
            Vector3 pos = transform.position;  // Boss’s coordinates at the firing timing

            // Repeat as many times as the number of spreads for the spread beam
            for (int i = 1; i <= SHOOT_NUM; i++)
            {
                // Generate beam (prefab_Beam2) in the scene (beam_A)
                GameObject beam_A = Instantiate(prefab_Beam2);
                // Move beam_A to the specified position
                beam_A.transform.position = new Vector3(pos.x + 3f, pos.y + 1f, 0);
                // Rotate the beam by 360° / number of spreads of the spread beam
                beam_A.transform.localEulerAngles = new Vector3(0, 0, i * 360 / SHOOT_NUM);

                // Generate beam (prefab_Beam2) in the scene (beam_B)
                GameObject beam_B = Instantiate(prefab_Beam2);
                // Move beam_B to the specified position
                beam_B.transform.position = new Vector3(pos.x - 2f, pos.y + 1f, 0);
                // Rotate the beam by 360° / number of spreads of the spread beam
                beam_B.transform.localEulerAngles = new Vector3(0, 0, i * 360 / SHOOT_NUM);

                // Generate beam (prefab_Beam2) in the scene (beam_C)
                GameObject beam_C = Instantiate(prefab_Beam2);
                // Move beam_C to the specified position
                beam_C.transform.position = new Vector3(pos.x + 3f, pos.y - 1f, 0);
                // Rotate the beam by 360° / number of spreads of the spread beam
                beam_C.transform.localEulerAngles = new Vector3(0, 0, i * 360 / SHOOT_NUM);

                // Generate beam (prefab_Beam2) in the scene (beam_D)
                GameObject beam_D = Instantiate(prefab_Beam2);
                // Move beam_D to the specified position
                beam_D.transform.position = new Vector3(pos.x - 2f, pos.y - 1f, 0);
                // Rotate the beam by 360° / number of spreads of the spread beam
                beam_D.transform.localEulerAngles = new Vector3(0, 0, i * 360 / SHOOT_NUM);
            }

            // Wait for the specified seconds
            yield return new WaitForSeconds(4.0f);

            // Call coroutine function Create_Beam2
            StartCoroutine(Create_Beam2());
        }
    }
}
