﻿using System.Collections;
using UnityEngine;

using UnityEngine.SceneManagement;  // Used for switching scenes
using UnityEngine.UI;  // Used for UI control processing

//////////////////////////////////////////////////////////////////
// [Class used for controlling the entire game] 
// Game start and restart processing, stage clear and game clear processing,
// continue and game over processing
//////////////////////////////////////////////////////////////////
public class GameController : MonoBehaviour
{
    /////////////////////////////////
    // [Variable settings]
    /////////////////////////////////

    // [Public variables accessed from other classes]
    public bool isGameFinish = false;  // Game end status (initially false)

    // [Variables set from the Unity Editor]
    [SerializeField] GameObject prefab_Ball;  // Prefab for generating a ball

    [SerializeField] GameObject go_Balls;  // Parent object of the balls
    [SerializeField] GameObject go_Blocks;  // Parent object of the blocks

    [SerializeField] GameObject go_GameOver;  // Text UI for game over
    [SerializeField] GameObject go_GameClear;  // Text UI for game clear
    [SerializeField] Text text_BallNum;  // Text UI for ball remaining number


    /////////////////////////////////
    // [Function settings]
    /////////////////////////////////

    // [Function that runs only once when the game object is generated]
    // (Unity standard function)
    void Start()
    {
        // Call the function for starting the stage
        StageStart();
    }


    // [Function that runs once every frame]
    // (Unity standard function)
    void Update()
    {
        // If the game has not ended, and all balls in the Scene are deleted
        if (isGameFinish == false && go_Balls.transform.childCount == 0)
        {
            // Call the function for game over
            GameOver();
        }

        // If the game has not ended, and all erasable blocks are deleted
        if (isGameFinish == false && go_Blocks.transform.childCount == 0)
        {
            // Call the coroutine function for game clear
            StartCoroutine(GameClear());
        }
    }


    // [Start processing function for game start and restart]
    void StageStart()
    {
        // Set game end status to false
        isGameFinish = false;
        // Hide the Game Over UI displayed at restart
        go_GameOver.SetActive(false);

        // Display the remaining number of balls in the UI text
        text_BallNum.text = SceneController.ballNum.ToString();

        // Generate one ball in the Scene
        GameObject ball = Instantiate(prefab_Ball,
            new Vector3(0, -3, 0), Quaternion.identity);
        // Move the generated ball under the parent object for balls
        ball.transform.parent = go_Balls.transform;
    }


    // [End processing function for continue and game over]
    public void GameOver()
    {
        // Set game end status to true
        isGameFinish = true;
        // Display the Game Over UI at continue and game over
        go_GameOver.SetActive(true);

        // If the remaining number of balls managed by SceneController > 0
        if (SceneController.ballNum > 0)
        {
            // Set the game over UI text to "Continue"
            go_GameOver.GetComponent<Text>().text = "Continue";

            // Decrease the remaining number of balls by 1
            SceneController.ballNum--;

            // Call the start processing function after a few seconds
            Invoke("StageStart", 3);
        }

        // If the remaining number of balls managed by SceneController is 0
        else
            // Set the game over UI text to "Game Over"
            go_GameOver.GetComponent<Text>().text = "Game Over";
    }


    // [Clear processing coroutine function called at stage clear or game clear]
    IEnumerator GameClear()
    {
        // Set game end status to true
        isGameFinish = true;
        // Display the Game Clear UI at stage clear or game clear
        go_GameClear.SetActive(true);

        // If the currently running scene is GameStage_1 or GameStage_2
        if (SceneManager.GetActiveScene().name == "GameStage_1"
            || SceneManager.GetActiveScene().name == "GameStage_2")
            // Set the game clear UI text to "Stage Clear"
            go_GameClear.GetComponent<Text>().text = "Stage Clear";

        // If the currently running scene is GameStage_3
        else
            // Set the game clear UI text to "Game Clear"
            go_GameClear.GetComponent<Text>().text = "Game Clear";

        // Display Game Clear or Stage Clear for a few seconds
        yield return new WaitForSeconds(3); // Wait for a few seconds

        // If the currently running scene is GameStage_1,
        if (SceneManager.GetActiveScene().name == "GameStage_1")
            // Launch the scene GameStage_2
            SceneManager.LoadScene("GameStage_2");

        // If the currently running scene is GameStage_2,
        else if (SceneManager.GetActiveScene().name == "GameStage_2")
            // Launch the scene GameStage_3
            SceneManager.LoadScene("GameStage_3");
    }
}
