﻿using UnityEngine;

//////////////////////////////////////////////////////////////////
// [Class used for controlling the ball]
// Initial speed control of the ball
// Speed control when the ball hits the Player bar, block, or wall
//////////////////////////////////////////////////////////////////
public class BallController : MonoBehaviour
{
    /////////////////////////////////
    // [Variable settings]
    /////////////////////////////////

    // [Variables set from the Unity Editor]
    [SerializeField] float ballSpeed = 4.0f;  // Speed of the ball
    // To check the speed of the ball from the Unity Editor side

    [SerializeField] Rigidbody2D rigidBody;  // Rigidbody2D component

    // [Private variables used in this program]
    GameController cl_GameController;  // GameController class


    /////////////////////////////////
    // [Function settings]
    /////////////////////////////////

    // [Function that runs only once when the game object is generated]
    // (Unity standard function)
    void Start()
    {
        // Obtain the GameController component from the GameController on the Scene
        cl_GameController
            = GameObject.Find("GameController").GetComponent<GameController>();

        // Random value to give the ball a random initial speed
        float randomValue = Random.value;  // Random value（0–1）

        // Initial direction of ball (Right: initDirect =1, Left: initDirect =-1)
        float initDirect = 1;  // Initial value of the ball’s direction: Right
        // If the random value < 0.5, initial value of the ball’s direction: Left
        if (randomValue < 0.5) initDirect = -1;

        // Give the ball a velocity using the initial speed and random value
        rigidBody.linearVelocity
            = new Vector3(ballSpeed * initDirect, ballSpeed + randomValue, 0f);
    }


    // [Function that runs once every frame]
    // (Unity standard function)
    void Update()
    {
        // Check GameController’s isGameFinish, and if the game has ended
        if (cl_GameController.isGameFinish)
        {
            // Delete the ball from the Scene
            Destroy(this.gameObject);
        }
    }


    // [Function called when the ball’s collider collides with another collider]
    // (Unity standard function)
    void OnCollisionEnter2D(Collision2D collision)
    {
        // When the ball hits the Player bar
        if (collision.gameObject.CompareTag("Player"))
        {
            // Get the coordinates of the ball center
            Vector3 ballPos = this.gameObject.transform.position;
            // Get the coordinates of the Player bar center
            Vector3 playerPos = collision.transform.position;

            // Get the direction of the ball center relative to the Player bar center
            Vector3 boundDirect = (ballPos - playerPos).normalized;

            // Give the ball a velocity vector in the direction of ball from Player bar
            rigidBody.linearVelocity = boundDirect * ballSpeed;
        }

        // When the ball hits a wall
        else if (collision.gameObject.CompareTag("Wall"))
        {
            // Get the current velocity of the ball
            Vector3 currentDirect = rigidBody.linearVelocity;

            // If the current vertical (Y-axis) speed is slower than 0.1
            if (Mathf.Abs(currentDirect.y) < 0.1)
            {
                // Force the downward speed to be faster (-0.5) (Important to come down)
                rigidBody.linearVelocity =
                    new Vector3(currentDirect.x, currentDirect.y - 0.5f, currentDirect.z);
            }
        }

        // When the ball hits the game over line at the bottom
        else if (collision.gameObject.CompareTag("Wall_B"))
        {
            // Delete the collided ball from the Scene
            Destroy(this.gameObject);
        }

        // When the ball hits a block
        else if (collision.gameObject.CompareTag("Block"))
        {
            // Get the unique additional speed that the hit block has
            float addSpeed =
                collision.gameObject.GetComponent<BlockController>().addSpeed;
            // Add the block’s additional speed to the ball’s speed value
            ballSpeed += addSpeed;

            // Give the ball a velocity vector with the added speed
            rigidBody.linearVelocity
                = rigidBody.linearVelocity.normalized * ballSpeed;
        }
    }
}
